<?php
require_once 'includes/config.php';

// Redirect to login if not authenticated
requireLogin();

// Check if user can view applications
if (!canViewApplications()) {
    header('Location: unauthorized.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();
$userAuth = new UserAuth($db);
$applicationForm = new ApplicationForm($db);
$exhibitionData = new InternationalExhibitionData($db);
$regionalData = new RegionalOfficesData($db);
$division = new Division($db);

// Get user data
$user = $userAuth->getUserById($_SESSION['user_id']);

// Handle status updates (Admin/Super Admin only)
if ($_POST && isset($_POST['update_status']) && ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin')) {
    $application_id = $_POST['application_id'];
    $status = $_POST['status'];
    $review_notes = $_POST['review_notes'];
    
    if ($applicationForm->updateApplicationStatus($application_id, $status, $_SESSION['user_id'], $review_notes)) {
        $userAuth->logActivity($_SESSION['user_id'], 'application_status_updated', 'Updated application status to: ' . $status);
        header('Location: applications.php?success=status_updated');
        exit();
    }
}

// Get applications based on user role
$applications = [];
if ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin' || $_SESSION['user_role'] === 'Moderator') {
    // Can see all applications
    $applications = $applicationForm->getAllApplications();
} else if ($_SESSION['user_role'] === 'Division Role') {
    // Can only see their own applications
    $applications = $applicationForm->getApplicationsByUser($_SESSION['user_id']);
}

// Get application stats
$stats = $applicationForm->getApplicationStats();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Applications - Dashboard</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>Dashboard</h2>
                <div class="user-info">
                    <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['user_name'], 0, 1)); ?></div>
                    <div class="user-details">
                        <div class="user-name"><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                        <div class="user-role"><?php echo htmlspecialchars($_SESSION['user_role']); ?></div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <a href="dashboard.php" class="nav-item">
                    <i class="icon">📊</i> Dashboard
                </a>
                <a href="applications.php" class="nav-item active">
                    <i class="icon">📝</i> Applications
                </a>
                
                <?php if (canCreateApplications()): ?>
                    <a href="international_exhibition.php" class="nav-item">
                        <i class="icon">🌍</i> International Exhibition
                    </a>
                    <a href="regional_offices.php" class="nav-item">
                        <i class="icon">🏢</i> Regional Offices
                    </a>
                <?php endif; ?>
                
                <a href="dashboard.php?logout=1" class="nav-item logout">
                    <i class="icon">🚪</i> Logout
                </a>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="content-header">
                <h1>Application Management</h1>
                <div class="header-actions">
                    <?php if (canCreateApplications()): ?>
                        <a href="international_exhibition.php" class="btn btn-primary">+ International Exhibition</a>
                        <a href="regional_offices.php" class="btn btn-primary">+ Regional Offices</a>
                    <?php endif; ?>
                </div>
            </div>
            
            <?php if (isset($_GET['success'])): ?>
                <div class="alert alert-success">
                    <?php 
                    switch($_GET['success']) {
                        case 'status_updated':
                            echo 'Application status updated successfully!';
                            break;
                    }
                    ?>
                </div>
            <?php endif; ?>
            
            <!-- Application Statistics -->
            <?php if ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin'): ?>
                <div class="card">
                    <div class="card-header">
                        <h3>📊 Application Statistics</h3>
                    </div>
                    <div class="card-content">
                        <div class="stats-grid">
                            <div class="stat-item">
                                <div class="stat-number"><?php echo $stats['total_applications']; ?></div>
                                <div class="stat-label">Total Applications</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-number pending"><?php echo $stats['pending']; ?></div>
                                <div class="stat-label">Pending Review</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-number approved"><?php echo $stats['approved']; ?></div>
                                <div class="stat-label">Approved</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-number rejected"><?php echo $stats['rejected']; ?></div>
                                <div class="stat-label">Rejected</div>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Applications List -->
            <div class="card">
                <div class="card-header">
                    <h3>📋 Applications</h3>
                </div>
                <div class="card-content">
                    <div class="applications-list">
                        <?php if (empty($applications)): ?>
                            <div class="empty-state">
                                <p>No applications found.</p>
                                <?php if (canCreateApplications()): ?>
                                    <p>Create your first application using the forms above!</p>
                                <?php endif; ?>
                            </div>
                        <?php else: ?>
                            <?php foreach ($applications as $application): ?>
                                <div class="application-item">
                                    <div class="application-info">
                                        <div class="application-header">
                                            <h4><?php echo htmlspecialchars($application['form_type']); ?></h4>
                                            <span class="application-id">#<?php echo $application['id']; ?></span>
                                        </div>
                                        
                                        <div class="application-details">
                                            <div class="detail-row">
                                                <span class="detail-label">Division:</span>
                                                <span class="detail-value"><?php echo htmlspecialchars($application['division_name']); ?> (<?php echo htmlspecialchars($application['division_code']); ?>)</span>
                                            </div>
                                            
                                            <div class="detail-row">
                                                <span class="detail-label">Submitted by:</span>
                                                <span class="detail-value"><?php echo htmlspecialchars($application['full_name']); ?> (<?php echo htmlspecialchars($application['email']); ?>)</span>
                                            </div>
                                            
                                            <div class="detail-row">
                                                <span class="detail-label">Submitted on:</span>
                                                <span class="detail-value"><?php echo date('M j, Y g:i A', strtotime($application['submitted_at'])); ?></span>
                                            </div>
                                            
                                            <?php if ($application['reviewed_at']): ?>
                                                <div class="detail-row">
                                                    <span class="detail-label">Reviewed on:</span>
                                                    <span class="detail-value"><?php echo date('M j, Y g:i A', strtotime($application['reviewed_at'])); ?></span>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                        
                                        <!-- Form-specific data -->
                                        <?php if ($application['form_type'] === 'International Exhibition'): ?>
                                            <?php 
                                            $exhibition_info = $exhibitionData->getExhibitionDataByApplication($application['id']);
                                            if ($exhibition_info): 
                                            ?>
                                                <div class="form-data">
                                                    <h5>Exhibition Details:</h5>
                                                    <div class="data-grid">
                                                        <div class="data-item">
                                                            <strong>Serial No:</strong> <?php echo htmlspecialchars($exhibition_info['serial_no']); ?>
                                                        </div>
                                                        <div class="data-item">
                                                            <strong>Activity:</strong> <?php echo htmlspecialchars($exhibition_info['activity_achievement']); ?>
                                                        </div>
                                                        <div class="data-item">
                                                            <strong>Date:</strong> <?php echo date('M j, Y', strtotime($exhibition_info['activity_date'])); ?>
                                                        </div>
                                                        <div class="data-item">
                                                            <strong>Description:</strong> <?php echo htmlspecialchars($exhibition_info['brief_description']); ?>
                                                        </div>
                                                        <div class="data-item">
                                                            <strong>Objective:</strong> <?php echo htmlspecialchars($exhibition_info['objective']); ?>
                                                        </div>
                                                        <div class="data-item">
                                                            <strong>Impact:</strong> <?php echo htmlspecialchars($exhibition_info['impact_significance_outcome']); ?>
                                                        </div>
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                            
                                        <?php elseif ($application['form_type'] === 'Regional Offices'): ?>
                                            <?php 
                                            $regional_info = $regionalData->getRegionalDataByApplication($application['id']);
                                            if ($regional_info): 
                                            ?>
                                                <div class="form-data">
                                                    <h5>Regional Details:</h5>
                                                    <div class="data-grid">
                                                        <div class="data-item">
                                                            <strong>Country:</strong> <?php echo htmlspecialchars($regional_info['country']); ?>
                                                        </div>
                                                        <div class="data-item">
                                                            <strong>Produce:</strong> <?php echo htmlspecialchars($regional_info['produce']); ?>
                                                        </div>
                                                        <?php if ($regional_info['date_wise']): ?>
                                                            <div class="data-item">
                                                                <strong>Date Wise:</strong> <?php echo date('M j, Y', strtotime($regional_info['date_wise'])); ?>
                                                            </div>
                                                        <?php endif; ?>
                                                        <?php if ($regional_info['year_wise']): ?>
                                                            <div class="data-item">
                                                                <strong>Year Wise:</strong> <?php echo $regional_info['year_wise']; ?>
                                                            </div>
                                                        <?php endif; ?>
                                                        <div class="data-item">
                                                            <strong>File Type:</strong> <?php echo htmlspecialchars($regional_info['file_type']); ?>
                                                        </div>
                                                        <?php if ($regional_info['file_path']): ?>
                                                            <div class="data-item">
                                                                <strong>File:</strong> 
                                                                <a href="<?php echo htmlspecialchars($regional_info['file_path']); ?>" target="_blank" class="file-link">
                                                                    View File
                                                                </a>
                                                            </div>
                                                        <?php endif; ?>
                                                    </div>
                                                </div>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                        
                                        <?php if ($application['review_notes']): ?>
                                            <div class="review-notes">
                                                <strong>Review Notes:</strong> <?php echo htmlspecialchars($application['review_notes']); ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    
                                    <div class="application-actions">
                                        <span class="status-badge status-<?php echo $application['status']; ?>">
                                            <?php echo ucfirst($application['status']); ?>
                                        </span>
                                        
                                        <?php if ($application['status'] === 'pending' && ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin')): ?>
                                            <form method="POST" class="status-form">
                                                <input type="hidden" name="application_id" value="<?php echo $application['id']; ?>">
                                                <select name="status" class="status-select">
                                                    <option value="approved">Approve</option>
                                                    <option value="rejected">Reject</option>
                                                </select>
                                                <textarea name="review_notes" placeholder="Review notes..." class="review-notes-input"></textarea>
                                                <button type="submit" name="update_status" class="btn btn-sm btn-primary">Update</button>
                                            </form>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="assets/js/dashboard.js"></script>
</body>
</html>
