// Dashboard JavaScript functionality
document.addEventListener('DOMContentLoaded', function() {
    
    // Initialize dashboard
    initializeDashboard();
    
    // Setup form handlers
    setupFormHandlers();
    
    // Setup interactive elements
    setupInteractiveElements();
    
    // Setup real-time updates
    setupRealTimeUpdates();
});

function initializeDashboard() {
    console.log('Dashboard initialized');
    
    // Add loading states to forms
    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const submitBtn = form.querySelector('button[type="submit"]');
            if (submitBtn) {
                submitBtn.innerHTML = 'Processing...';
                submitBtn.disabled = true;
            }
        });
    });
    
    // Auto-hide alerts after 5 seconds
    const alerts = document.querySelectorAll('.alert');
    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.opacity = '0';
            setTimeout(() => {
                alert.remove();
            }, 300);
        }, 5000);
    });
}

function setupFormHandlers() {
    // Submission form validation
    const submissionForm = document.querySelector('.submission-form');
    if (submissionForm) {
        submissionForm.addEventListener('submit', function(e) {
            const title = document.getElementById('title');
            const description = document.getElementById('description');
            
            if (!title.value.trim()) {
                e.preventDefault();
                showAlert('Please enter a title', 'error');
                title.focus();
                return;
            }
            
            if (!description.value.trim()) {
                e.preventDefault();
                showAlert('Please enter a description', 'error');
                description.focus();
                return;
            }
            
            if (description.value.trim().length < 10) {
                e.preventDefault();
                showAlert('Description must be at least 10 characters long', 'error');
                description.focus();
                return;
            }
        });
    }
    
    // Status update form validation
    const statusForms = document.querySelectorAll('.status-form');
    statusForms.forEach(form => {
        form.addEventListener('submit', function(e) {
            const status = form.querySelector('select[name="status"]');
            const reviewNotes = form.querySelector('textarea[name="review_notes"]');
            
            if (status.value === 'rejected' && !reviewNotes.value.trim()) {
                e.preventDefault();
                showAlert('Please provide review notes for rejected submissions', 'error');
                reviewNotes.focus();
                return;
            }
        });
    });
}

function setupInteractiveElements() {
    // Smooth scrolling for navigation
    const navItems = document.querySelectorAll('.nav-item[href^="#"]');
    navItems.forEach(item => {
        item.addEventListener('click', function(e) {
            e.preventDefault();
            const targetId = this.getAttribute('href').substring(1);
            const targetElement = document.getElementById(targetId);
            
            if (targetElement) {
                targetElement.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
    
    // Status badge animations
    const statusBadges = document.querySelectorAll('.status-badge');
    statusBadges.forEach(badge => {
        badge.addEventListener('mouseenter', function() {
            this.style.transform = 'scale(1.05)';
        });
        
        badge.addEventListener('mouseleave', function() {
            this.style.transform = 'scale(1)';
        });
    });
    
    // Card hover effects
    const cards = document.querySelectorAll('.card');
    cards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
            this.style.boxShadow = '0 10px 25px rgba(0, 0, 0, 0.15)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
            this.style.boxShadow = '0 5px 15px rgba(0, 0, 0, 0.08)';
        });
    });
    
    // Character counter for textareas
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
        const maxLength = textarea.getAttribute('maxlength') || 500;
        const counter = document.createElement('div');
        counter.className = 'char-counter';
        counter.style.cssText = 'font-size: 12px; color: #6c757d; text-align: right; margin-top: 5px;';
        
        textarea.parentNode.appendChild(counter);
        
        function updateCounter() {
            const remaining = maxLength - textarea.value.length;
            counter.textContent = `${textarea.value.length}/${maxLength} characters`;
            
            if (remaining < 50) {
                counter.style.color = '#dc3545';
            } else if (remaining < 100) {
                counter.style.color = '#ffc107';
            } else {
                counter.style.color = '#6c757d';
            }
        }
        
        textarea.addEventListener('input', updateCounter);
        updateCounter();
    });
}

function setupRealTimeUpdates() {
    // Auto-refresh submission list every 30 seconds (for admin users)
    const submissionList = document.querySelector('.submissions-list');
    if (submissionList && window.location.pathname.includes('dashboard.php')) {
        setInterval(() => {
            refreshSubmissions();
        }, 30000);
    }
    
    // Real-time status updates
    const statusSelects = document.querySelectorAll('.status-select');
    statusSelects.forEach(select => {
        select.addEventListener('change', function() {
            const form = this.closest('form');
            const reviewNotes = form.querySelector('.review-notes');
            
            if (this.value === 'rejected') {
                reviewNotes.required = true;
                reviewNotes.placeholder = 'Please explain why this submission is being rejected...';
            } else {
                reviewNotes.required = false;
                reviewNotes.placeholder = 'Review notes (optional)...';
            }
        });
    });
}

function refreshSubmissions() {
    // This would typically make an AJAX call to refresh the submissions
    // For now, we'll just show a subtle indicator
    const refreshIndicator = document.createElement('div');
    refreshIndicator.className = 'refresh-indicator';
    refreshIndicator.innerHTML = '🔄 Refreshing...';
    refreshIndicator.style.cssText = 'position: fixed; top: 20px; right: 20px; background: #667eea; color: white; padding: 10px 15px; border-radius: 5px; z-index: 1000; font-size: 12px;';
    
    document.body.appendChild(refreshIndicator);
    
    setTimeout(() => {
        refreshIndicator.remove();
    }, 2000);
}

function showAlert(message, type = 'info') {
    const alert = document.createElement('div');
    alert.className = `alert alert-${type}`;
    alert.textContent = message;
    
    // Insert at the top of main content
    const mainContent = document.querySelector('.main-content');
    const firstChild = mainContent.firstChild;
    mainContent.insertBefore(alert, firstChild);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        alert.style.opacity = '0';
        setTimeout(() => {
            alert.remove();
        }, 300);
    }, 5000);
}

// Utility functions
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    });
}

function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Search functionality (if needed)
function setupSearch() {
    const searchInput = document.getElementById('search');
    if (searchInput) {
        const debouncedSearch = debounce(function(e) {
            const query = e.target.value.toLowerCase();
            const submissionItems = document.querySelectorAll('.submission-item');
            
            submissionItems.forEach(item => {
                const title = item.querySelector('h4').textContent.toLowerCase();
                const description = item.querySelector('p').textContent.toLowerCase();
                
                if (title.includes(query) || description.includes(query)) {
                    item.style.display = 'flex';
                } else {
                    item.style.display = 'none';
                }
            });
        }, 300);
        
        searchInput.addEventListener('input', debouncedSearch);
    }
}

// Export functions for potential use in other scripts
window.DashboardUtils = {
    showAlert,
    formatDate,
    debounce
};
