<?php
require_once 'includes/config.php';

// Logout functionality
if (isset($_GET['logout'])) {
    if (isLoggedIn()) {
        $database = new Database();
        $db = $database->getConnection();
        $userAuth = new UserAuth($db);
        $userAuth->logActivity($_SESSION['user_id'], 'logout', 'User logged out');
    }
    
    session_destroy();
    header('Location: login.php');
    exit();
}

// Redirect to login if not authenticated
requireLogin();

$database = new Database();
$db = $database->getConnection();
$userAuth = new UserAuth($db);
$workSubmission = new WorkSubmission($db);
$applicationForm = new ApplicationForm($db);
$division = new Division($db);

// Get user data
$user = $userAuth->getUserById($_SESSION['user_id']);

// Get dashboard data based on role
$dashboard_data = [];
$submissions = [];
$applications = [];

if ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin') {
    // Admin can see all submissions and applications
    $submissions = $workSubmission->getAllSubmissions();
    $applications = $applicationForm->getAllApplications();
    $stats = $workSubmission->getSubmissionStats();
    $app_stats = $applicationForm->getApplicationStats();
} else {
    // Regular users see only their submissions and applications
    $submissions = $workSubmission->getSubmissionsByUser($_SESSION['user_id']);
    $applications = $applicationForm->getApplicationsByUser($_SESSION['user_id']);
}

// Get user's division
$user_division = null;
if ($_SESSION['division_id']) {
    $user_division = $division->getDivisionById($_SESSION['division_id']);
}

// Handle form submissions
if ($_POST && ($_SESSION['user_role'] === 'Division Role' || $_SESSION['user_role'] === 'Moderator')) {
    $title = $_POST['title'];
    $description = $_POST['description'];
    $submission_type = $_POST['submission_type'];
    
    if ($workSubmission->createSubmission($_SESSION['user_id'], $title, $description, $submission_type)) {
        $userAuth->logActivity($_SESSION['user_id'], 'submission_created', 'Created new work submission: ' . $title);
        header('Location: dashboard.php?success=submission_created');
        exit();
    }
}

// Handle status updates (Admin/Super Admin only)
if ($_POST && isset($_POST['update_status']) && ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin')) {
    $submission_id = $_POST['submission_id'];
    $status = $_POST['status'];
    $review_notes = $_POST['review_notes'];
    
    if ($workSubmission->updateSubmissionStatus($submission_id, $status, $_SESSION['user_id'], $review_notes)) {
        $userAuth->logActivity($_SESSION['user_id'], 'status_updated', 'Updated submission status to: ' . $status);
        header('Location: dashboard.php?success=status_updated');
        exit();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - <?php echo htmlspecialchars($_SESSION['user_role']); ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>Dashboard</h2>
                <div class="user-info">
                    <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['user_name'], 0, 1)); ?></div>
                    <div class="user-details">
                        <div class="user-name"><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                        <div class="user-role"><?php echo htmlspecialchars($_SESSION['user_role']); ?></div>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <a href="dashboard.php" class="nav-item active">
                    <i class="icon">📊</i> Dashboard
                </a>
                
                <a href="applications.php" class="nav-item">
                    <i class="icon">📝</i> Applications
                </a>
                
                <?php if (canCreateApplications()): ?>
                    <a href="international_exhibition.php" class="nav-item">
                        <i class="icon">🌍</i> International Exhibition
                    </a>
                    <a href="regional_offices.php" class="nav-item">
                        <i class="icon">🏢</i> Regional Offices
                    </a>
                <?php endif; ?>
                
                <?php if ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin'): ?>
                    <a href="#admin-panel" class="nav-item">
                        <i class="icon">⚙️</i> Admin Panel
                    </a>
                    <a href="#all-submissions" class="nav-item">
                        <i class="icon">📋</i> All Submissions
                    </a>
                <?php endif; ?>
                
                <a href="dashboard.php?logout=1" class="nav-item logout">
                    <i class="icon">🚪</i> Logout
                </a>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="content-header">
                <h1>Welcome, <?php echo htmlspecialchars($_SESSION['user_name']); ?>!</h1>
                <div class="header-actions">
                    <span class="last-login">Last login: <?php echo $user['last_login'] ? date('M j, Y g:i A', strtotime($user['last_login'])) : 'First time'; ?></span>
                </div>
            </div>
            
            <?php if (isset($_GET['success'])): ?>
                <div class="alert alert-success">
                    <?php 
                    switch($_GET['success']) {
                        case 'submission_created':
                            echo 'Work submission created successfully!';
                            break;
                        case 'status_updated':
                            echo 'Submission status updated successfully!';
                            break;
                    }
                    ?>
                </div>
            <?php endif; ?>
            
            <!-- Dashboard Cards -->
            <div class="dashboard-cards">
                <?php if ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin'): ?>
                    <!-- Admin Dashboard -->
                    <div class="card">
                        <div class="card-header">
                            <h3>📊 Application Statistics</h3>
                        </div>
                        <div class="card-content">
                            <div class="stats-grid">
                                <div class="stat-item">
                                    <div class="stat-number"><?php echo $app_stats['total_applications']; ?></div>
                                    <div class="stat-label">Total Applications</div>
                                </div>
                                <div class="stat-item">
                                    <div class="stat-number pending"><?php echo $app_stats['pending']; ?></div>
                                    <div class="stat-label">Pending Review</div>
                                </div>
                                <div class="stat-item">
                                    <div class="stat-number approved"><?php echo $app_stats['approved']; ?></div>
                                    <div class="stat-label">Approved</div>
                                </div>
                                <div class="stat-item">
                                    <div class="stat-number rejected"><?php echo $app_stats['rejected']; ?></div>
                                    <div class="stat-label">Rejected</div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card">
                        <div class="card-header">
                            <h3>📋 All Applications</h3>
                        </div>
                        <div class="card-content">
                            <div class="applications-list">
                                <?php foreach ($applications as $application): ?>
                                    <div class="application-item">
                                        <div class="application-info">
                                            <h4><?php echo htmlspecialchars($application['form_type']); ?></h4>
                                            <p><strong>Division:</strong> <?php echo htmlspecialchars($application['division_name']); ?></p>
                                            <p><strong>Submitted by:</strong> <?php echo htmlspecialchars($application['full_name']); ?></p>
                                            <p><strong>Date:</strong> <?php echo date('M j, Y', strtotime($application['submitted_at'])); ?></p>
                                        </div>
                                        <div class="application-actions">
                                            <span class="status-badge status-<?php echo $application['status']; ?>">
                                                <?php echo ucfirst($application['status']); ?>
                                            </span>
                                            
                                            <?php if ($application['status'] === 'pending'): ?>
                                                <form method="POST" class="status-form">
                                                    <input type="hidden" name="application_id" value="<?php echo $application['id']; ?>">
                                                    <select name="status" class="status-select">
                                                        <option value="approved">Approve</option>
                                                        <option value="rejected">Reject</option>
                                                    </select>
                                                    <textarea name="review_notes" placeholder="Review notes..." class="review-notes"></textarea>
                                                    <button type="submit" name="update_status" class="btn btn-sm btn-primary">Update</button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                    
                <?php else: ?>
                    <!-- Regular User Dashboard -->
                    <div class="card">
                        <div class="card-header">
                            <h3>📝 My Applications</h3>
                        </div>
                        <div class="card-content">
                            <div class="applications-list">
                                <?php if (empty($applications)): ?>
                                    <div class="empty-state">
                                        <p>No applications yet. Create your first application using the forms above!</p>
                                    </div>
                                <?php else: ?>
                                    <?php foreach ($applications as $application): ?>
                                        <div class="application-item">
                                            <div class="application-info">
                                                <h4><?php echo htmlspecialchars($application['form_type']); ?></h4>
                                                <p><strong>Division:</strong> <?php echo htmlspecialchars($application['division_name']); ?></p>
                                                <p><strong>Date:</strong> <?php echo date('M j, Y', strtotime($application['submitted_at'])); ?></p>
                                            </div>
                                            <div class="application-status">
                                                <span class="status-badge status-<?php echo $application['status']; ?>">
                                                    <?php echo ucfirst($application['status']); ?>
                                                </span>
                                                <?php if ($application['review_notes']): ?>
                                                    <div class="review-notes">
                                                        <strong>Review Notes:</strong> <?php echo htmlspecialchars($application['review_notes']); ?>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Application Forms -->
                    <?php if (canCreateApplications()): ?>
                        <div class="card">
                            <div class="card-header">
                                <h3>➕ Create New Application</h3>
                            </div>
                            <div class="card-content">
                                <div class="form-options">
                                    <a href="international_exhibition.php" class="form-option">
                                        <div class="form-icon">🌍</div>
                                        <h4>International Exhibition</h4>
                                        <p>Submit international exhibition activities and achievements</p>
                                    </a>
                                    <a href="regional_offices.php" class="form-option">
                                        <div class="form-icon">🏢</div>
                                        <h4>Regional Offices</h4>
                                        <p>Submit regional office activities and data</p>
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script src="assets/js/dashboard.js"></script>
</body>
</html>
