<?php
// Database configuration
class Database {
    private $host = 'localhost';
    private $db_name = 'dashboardtdapgov_dashboard_db';
    private $username = 'dashboardtdapgov_dashboard_user';
    private $password = 'iS^G0ydT#PX8c[OT';
    private $conn;

    public function getConnection() {
        $this->conn = null;
        
        try {
            $this->conn = new PDO(
                "mysql:host=" . $this->host . ";dbname=" . $this->db_name,
                $this->username,
                $this->password
            );
            $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        } catch(PDOException $exception) {
            echo "Connection error: " . $exception->getMessage();
        }
        
        return $this->conn;
    }
}

// User authentication class
class UserAuth {
    private $conn;
    private $table_name = "users";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function login($email, $password) {
        $query = "SELECT id, email, password, role, full_name FROM " . $this->table_name . " 
                  WHERE email = :email AND is_active = 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':email', $email);
        $stmt->execute();
        
        if($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            if(password_verify($password, $row['password'])) {
                // Update last login
                $this->updateLastLogin($row['id']);
                
                // Log activity
                $this->logActivity($row['id'], 'login', 'User logged in');
                
                return $row;
            }
        }
        return false;
    }

    public function updateLastLogin($user_id) {
        $query = "UPDATE " . $this->table_name . " SET last_login = NOW() WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->execute();
    }

    public function logActivity($user_id, $action, $description = '') {
        $query = "INSERT INTO activity_logs (user_id, action, description, ip_address) 
                  VALUES (:user_id, :action, :description, :ip_address)";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':action', $action);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':ip_address', $_SERVER['REMOTE_ADDR']);
        $stmt->execute();
    }

    public function getUserById($user_id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id = :user_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// Division management class
class Division {
    private $conn;
    private $table_name = "divisions";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function getAllDivisions() {
        $query = "SELECT * FROM " . $this->table_name . " ORDER BY name";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getDivisionById($division_id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id = :division_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':division_id', $division_id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function getDivisionByCode($code) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE code = :code";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':code', $code);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// Application form management class
class ApplicationForm {
    private $conn;
    private $table_name = "application_forms";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function createApplication($user_id, $division_id, $form_type) {
        $query = "INSERT INTO " . $this->table_name . " 
                  (user_id, division_id, form_type) 
                  VALUES (:user_id, :division_id, :form_type)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':division_id', $division_id);
        $stmt->bindParam(':form_type', $form_type);
        
        if ($stmt->execute()) {
            return $this->conn->lastInsertId();
        }
        return false;
    }

    public function getApplicationsByUser($user_id) {
        $query = "SELECT af.*, d.name as division_name, d.code as division_code 
                  FROM " . $this->table_name . " af 
                  JOIN divisions d ON af.division_id = d.id 
                  WHERE af.user_id = :user_id 
                  ORDER BY af.submitted_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getApplicationsByDivision($division_id) {
        $query = "SELECT af.*, d.name as division_name, d.code as division_code, u.full_name, u.email 
                  FROM " . $this->table_name . " af 
                  JOIN divisions d ON af.division_id = d.id 
                  JOIN users u ON af.user_id = u.id 
                  WHERE af.division_id = :division_id 
                  ORDER BY af.submitted_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':division_id', $division_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getAllApplications() {
        $query = "SELECT af.*, d.name as division_name, d.code as division_code, u.full_name, u.email 
                  FROM " . $this->table_name . " af 
                  JOIN divisions d ON af.division_id = d.id 
                  JOIN users u ON af.user_id = u.id 
                  ORDER BY af.submitted_at DESC";
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function updateApplicationStatus($application_id, $status, $reviewed_by, $review_notes = '') {
        $query = "UPDATE " . $this->table_name . " 
                  SET status = :status, reviewed_by = :reviewed_by, 
                      reviewed_at = NOW(), review_notes = :review_notes 
                  WHERE id = :application_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':status', $status);
        $stmt->bindParam(':reviewed_by', $reviewed_by);
        $stmt->bindParam(':review_notes', $review_notes);
        $stmt->bindParam(':application_id', $application_id);
        
        return $stmt->execute();
    }

    public function getApplicationStats() {
        $query = "SELECT 
                    COUNT(*) as total_applications,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                    SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
                  FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// International Exhibition data management
class InternationalExhibitionData {
    private $conn;
    private $table_name = "international_exhibition_data";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function createExhibitionData($application_id, $serial_no, $activity_achievement, $activity_date, $brief_description, $objective, $impact_significance_outcome) {
        $query = "INSERT INTO " . $this->table_name . " 
                  (application_id, serial_no, activity_achievement, activity_date, brief_description, objective, impact_significance_outcome) 
                  VALUES (:application_id, :serial_no, :activity_achievement, :activity_date, :brief_description, :objective, :impact_significance_outcome)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':application_id', $application_id);
        $stmt->bindParam(':serial_no', $serial_no);
        $stmt->bindParam(':activity_achievement', $activity_achievement);
        $stmt->bindParam(':activity_date', $activity_date);
        $stmt->bindParam(':brief_description', $brief_description);
        $stmt->bindParam(':objective', $objective);
        $stmt->bindParam(':impact_significance_outcome', $impact_significance_outcome);
        
        return $stmt->execute();
    }

    public function getExhibitionDataByApplication($application_id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE application_id = :application_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':application_id', $application_id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// Regional Offices data management
class RegionalOfficesData {
    private $conn;
    private $table_name = "regional_offices_data";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function createRegionalData($application_id, $country, $produce, $date_wise, $year_wise, $file_path, $file_type) {
        $query = "INSERT INTO " . $this->table_name . " 
                  (application_id, country, produce, date_wise, year_wise, file_path, file_type) 
                  VALUES (:application_id, :country, :produce, :date_wise, :year_wise, :file_path, :file_type)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':application_id', $application_id);
        $stmt->bindParam(':country', $country);
        $stmt->bindParam(':produce', $produce);
        $stmt->bindParam(':date_wise', $date_wise);
        $stmt->bindParam(':year_wise', $year_wise);
        $stmt->bindParam(':file_path', $file_path);
        $stmt->bindParam(':file_type', $file_type);
        
        return $stmt->execute();
    }

    public function getRegionalDataByApplication($application_id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE application_id = :application_id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':application_id', $application_id);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// Work submission class (keeping for backward compatibility)
class WorkSubmission {
    private $conn;
    private $table_name = "work_submissions";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function createSubmission($user_id, $title, $description, $submission_type = 'other') {
        $query = "INSERT INTO " . $this->table_name . " 
                  (user_id, title, description, submission_type) 
                  VALUES (:user_id, :title, :description, :submission_type)";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->bindParam(':title', $title);
        $stmt->bindParam(':description', $description);
        $stmt->bindParam(':submission_type', $submission_type);
        
        return $stmt->execute();
    }

    public function getSubmissionsByUser($user_id) {
        $query = "SELECT * FROM " . $this->table_name . " WHERE user_id = :user_id ORDER BY submitted_at DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':user_id', $user_id);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getAllSubmissions() {
        $query = "SELECT ws.*, u.full_name, u.email FROM " . $this->table_name . " ws 
                  JOIN users u ON ws.user_id = u.id 
                  ORDER BY ws.submitted_at DESC";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function updateSubmissionStatus($submission_id, $status, $reviewed_by, $review_notes = '') {
        $query = "UPDATE " . $this->table_name . " 
                  SET status = :status, reviewed_by = :reviewed_by, 
                      reviewed_at = NOW(), review_notes = :review_notes 
                  WHERE id = :submission_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':status', $status);
        $stmt->bindParam(':reviewed_by', $reviewed_by);
        $stmt->bindParam(':review_notes', $review_notes);
        $stmt->bindParam(':submission_id', $submission_id);
        
        return $stmt->execute();
    }

    public function getSubmissionStats() {
        $query = "SELECT 
                    COUNT(*) as total_submissions,
                    SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                    SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                    SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected
                  FROM " . $this->table_name;
        
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
}

// Check if user is logged in
function isLoggedIn() {
    return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
}

// Check user role
function hasRole($required_role) {
    if (!isLoggedIn()) return false;
    
    $user_role = $_SESSION['user_role'];
    $role_hierarchy = ['Moderator' => 1, 'Division Role' => 2, 'Admin' => 3, 'Super Admin' => 4];
    
    return isset($role_hierarchy[$user_role]) && 
           isset($role_hierarchy[$required_role]) && 
           $role_hierarchy[$user_role] >= $role_hierarchy[$required_role];
}

// Check if user can access division data
function canAccessDivision($division_id) {
    if (!isLoggedIn()) return false;
    
    // Super Admin and Admin can access all divisions
    if ($_SESSION['user_role'] === 'Super Admin' || $_SESSION['user_role'] === 'Admin') {
        return true;
    }
    
    // Moderator can access all divisions
    if ($_SESSION['user_role'] === 'Moderator') {
        return true;
    }
    
    // Division Role users can only access their own division
    if ($_SESSION['user_role'] === 'Division Role') {
        return isset($_SESSION['division_id']) && $_SESSION['division_id'] == $division_id;
    }
    
    return false;
}

// Check if user can view applications
function canViewApplications() {
    if (!isLoggedIn()) return false;
    
    return $_SESSION['user_role'] === 'Super Admin' || 
           $_SESSION['user_role'] === 'Admin' || 
           $_SESSION['user_role'] === 'Moderator' || 
           $_SESSION['user_role'] === 'Division Role';
}

// Check if user can create applications
function canCreateApplications() {
    if (!isLoggedIn()) return false;
    
    return $_SESSION['user_role'] === 'Division Role' || $_SESSION['user_role'] === 'Moderator';
}

// Redirect if not logged in
function requireLogin() {
    if (!isLoggedIn()) {
        header('Location: login.php');
        exit();
    }
}

// Redirect if doesn't have required role
function requireRole($role) {
    requireLogin();
    if (!hasRole($role)) {
        header('Location: unauthorized.php');
        exit();
    }
}
?>
