<?php
require_once 'includes/config.php';

// Redirect to login if not authenticated
requireLogin();

// Check if user can create applications
if (!canCreateApplications()) {
    header('Location: unauthorized.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();
$userAuth = new UserAuth($db);
$applicationForm = new ApplicationForm($db);
$exhibitionData = new InternationalExhibitionData($db);
$division = new Division($db);

// Get user data
$user = $userAuth->getUserById($_SESSION['user_id']);

// Handle form submission
if ($_POST && isset($_POST['submit_exhibition'])) {
    $serial_no = $_POST['serial_no'];
    $activity_achievement = $_POST['activity_achievement'];
    $activity_date = $_POST['activity_date'];
    $brief_description = $_POST['brief_description'];
    $objective = $_POST['objective'];
    $impact_significance_outcome = $_POST['impact_significance_outcome'];
    
    // Create application
    $application_id = $applicationForm->createApplication($_SESSION['user_id'], $_SESSION['division_id'], 'International Exhibition');
    
    if ($application_id) {
        // Create exhibition data
        if ($exhibitionData->createExhibitionData($application_id, $serial_no, $activity_achievement, $activity_date, $brief_description, $objective, $impact_significance_outcome)) {
            $userAuth->logActivity($_SESSION['user_id'], 'exhibition_application_created', 'Created International Exhibition application: ' . $activity_achievement);
            header('Location: dashboard.php?success=exhibition_created');
            exit();
        }
    }
}

// Get user's division
$user_division = null;
if ($_SESSION['division_id']) {
    $user_division = $division->getDivisionById($_SESSION['division_id']);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>International Exhibition Form - Dashboard</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>Dashboard</h2>
                <div class="user-info">
                    <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['user_name'], 0, 1)); ?></div>
                    <div class="user-details">
                        <div class="user-name"><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                        <div class="user-role"><?php echo htmlspecialchars($_SESSION['user_role']); ?></div>
                        <?php if ($user_division): ?>
                            <div class="user-division"><?php echo htmlspecialchars($user_division['name']); ?></div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <a href="dashboard.php" class="nav-item">
                    <i class="icon">📊</i> Dashboard
                </a>
                <a href="applications.php" class="nav-item">
                    <i class="icon">📝</i> My Applications
                </a>
                <a href="international_exhibition.php" class="nav-item active">
                    <i class="icon">🌍</i> International Exhibition
                </a>
                <a href="regional_offices.php" class="nav-item">
                    <i class="icon">🏢</i> Regional Offices
                </a>
                <a href="dashboard.php?logout=1" class="nav-item logout">
                    <i class="icon">🚪</i> Logout
                </a>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="content-header">
                <h1>International Exhibition Application Form</h1>
                <div class="header-actions">
                    <a href="dashboard.php" class="btn btn-secondary">← Back to Dashboard</a>
                </div>
            </div>
            
            <div class="dashboard-cards">
                <div class="card">
                    <div class="card-header">
                        <h3>🌍 International Exhibition Application</h3>
                        <p>Submit your international exhibition activities and achievements</p>
                    </div>
                    <div class="card-content">
                        <form method="POST" class="application-form" id="exhibitionForm">
                            <div class="form-section">
                                <h4>Activity Information</h4>
                                
                                <div class="form-row">
                                    <div class="form-group">
                                        <label for="serial_no">Serial No *</label>
                                        <input type="text" id="serial_no" name="serial_no" required 
                                               placeholder="Enter serial number" 
                                               value="<?php echo isset($_POST['serial_no']) ? htmlspecialchars($_POST['serial_no']) : ''; ?>">
                                    </div>
                                    
                                    <div class="form-group">
                                        <label for="activity_date">Date *</label>
                                        <input type="date" id="activity_date" name="activity_date" required 
                                               value="<?php echo isset($_POST['activity_date']) ? $_POST['activity_date'] : date('Y-m-d'); ?>">
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <label for="activity_achievement">Activity/Achievement *</label>
                                    <input type="text" id="activity_achievement" name="activity_achievement" required 
                                           placeholder="Enter activity or achievement title" 
                                           value="<?php echo isset($_POST['activity_achievement']) ? htmlspecialchars($_POST['activity_achievement']) : ''; ?>">
                                </div>
                                
                                <div class="form-group">
                                    <label for="brief_description">Brief Description *</label>
                                    <textarea id="brief_description" name="brief_description" required 
                                              placeholder="Provide a brief description of the activity" 
                                              rows="4"><?php echo isset($_POST['brief_description']) ? htmlspecialchars($_POST['brief_description']) : ''; ?></textarea>
                                </div>
                                
                                <div class="form-group">
                                    <label for="objective">Objective *</label>
                                    <textarea id="objective" name="objective" required 
                                              placeholder="Describe the objective of this activity" 
                                              rows="4"><?php echo isset($_POST['objective']) ? htmlspecialchars($_POST['objective']) : ''; ?></textarea>
                                </div>
                                
                                <div class="form-group">
                                    <label for="impact_significance_outcome">Impact/Significance/Outcome *</label>
                                    <textarea id="impact_significance_outcome" name="impact_significance_outcome" required 
                                              placeholder="Describe the impact, significance, or outcome of this activity" 
                                              rows="4"><?php echo isset($_POST['impact_significance_outcome']) ? htmlspecialchars($_POST['impact_significance_outcome']) : ''; ?></textarea>
                                </div>
                            </div>
                            
                            <div class="form-actions">
                                <button type="submit" name="submit_exhibition" class="btn btn-primary">
                                    Submit Application
                                </button>
                                <button type="button" class="btn btn-secondary" onclick="resetForm()">
                                    Reset Form
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Form Guidelines -->
                <div class="card">
                    <div class="card-header">
                        <h3>📋 Form Guidelines</h3>
                    </div>
                    <div class="card-content">
                        <div class="guidelines">
                            <h4>Please ensure:</h4>
                            <ul>
                                <li>All fields marked with * are required</li>
                                <li>Serial number should be unique for your division</li>
                                <li>Date should be the actual date of the activity</li>
                                <li>Brief description should be concise but informative</li>
                                <li>Objective should clearly state the purpose</li>
                                <li>Impact section should highlight measurable outcomes</li>
                            </ul>
                            
                            <h4>Division Information:</h4>
                            <?php if ($user_division): ?>
                                <p><strong>Your Division:</strong> <?php echo htmlspecialchars($user_division['name']); ?> (<?php echo htmlspecialchars($user_division['code']); ?>)</p>
                                <p><strong>Description:</strong> <?php echo htmlspecialchars($user_division['description']); ?></p>
                            <?php else: ?>
                                <p class="alert alert-error">No division assigned. Please contact administrator.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="assets/js/dashboard.js"></script>
    <script>
        function resetForm() {
            if (confirm('Are you sure you want to reset the form? All data will be lost.')) {
                document.getElementById('exhibitionForm').reset();
            }
        }
        
        // Form validation
        document.getElementById('exhibitionForm').addEventListener('submit', function(e) {
            const serialNo = document.getElementById('serial_no').value.trim();
            const activityAchievement = document.getElementById('activity_achievement').value.trim();
            const briefDescription = document.getElementById('brief_description').value.trim();
            const objective = document.getElementById('objective').value.trim();
            const impact = document.getElementById('impact_significance_outcome').value.trim();
            
            if (!serialNo || !activityAchievement || !briefDescription || !objective || !impact) {
                e.preventDefault();
                alert('Please fill in all required fields.');
                return;
            }
            
            if (briefDescription.length < 20) {
                e.preventDefault();
                alert('Brief description must be at least 20 characters long.');
                document.getElementById('brief_description').focus();
                return;
            }
            
            if (objective.length < 20) {
                e.preventDefault();
                alert('Objective must be at least 20 characters long.');
                document.getElementById('objective').focus();
                return;
            }
            
            if (impact.length < 20) {
                e.preventDefault();
                alert('Impact/Significance/Outcome must be at least 20 characters long.');
                document.getElementById('impact_significance_outcome').focus();
                return;
            }
        });
    </script>
</body>
</html>
