<?php
require_once 'includes/config.php';

// Redirect to login if not authenticated
requireLogin();

// Check if user can create applications
if (!canCreateApplications()) {
    header('Location: unauthorized.php');
    exit();
}

$database = new Database();
$db = $database->getConnection();
$userAuth = new UserAuth($db);
$applicationForm = new ApplicationForm($db);
$regionalData = new RegionalOfficesData($db);
$division = new Division($db);

// Get user data
$user = $userAuth->getUserById($_SESSION['user_id']);

// Handle form submission
if ($_POST && isset($_POST['submit_regional'])) {
    $country = $_POST['country'];
    $produce = $_POST['produce'];
    $date_wise = $_POST['date_wise'] ?: null;
    $year_wise = $_POST['year_wise'] ?: null;
    $file_type = $_POST['file_type'];
    
    // Handle file upload
    $file_path = null;
    if (isset($_FILES['upload_file']) && $_FILES['upload_file']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = 'uploads/';
        if (!file_exists($upload_dir)) {
            mkdir($upload_dir, 0777, true);
        }
        
        $file_extension = strtolower(pathinfo($_FILES['upload_file']['name'], PATHINFO_EXTENSION));
        $allowed_extensions = ['csv', 'pdf'];
        
        if (in_array($file_extension, $allowed_extensions)) {
            $file_name = uniqid() . '_' . time() . '.' . $file_extension;
            $file_path = $upload_dir . $file_name;
            
            if (move_uploaded_file($_FILES['upload_file']['tmp_name'], $file_path)) {
                $file_type = strtoupper($file_extension);
            } else {
                $file_path = null;
            }
        }
    }
    
    // Create application
    $application_id = $applicationForm->createApplication($_SESSION['user_id'], $_SESSION['division_id'], 'Regional Offices');
    
    if ($application_id) {
        // Create regional data
        if ($regionalData->createRegionalData($application_id, $country, $produce, $date_wise, $year_wise, $file_path, $file_type)) {
            $userAuth->logActivity($_SESSION['user_id'], 'regional_application_created', 'Created Regional Offices application for: ' . $country);
            header('Location: dashboard.php?success=regional_created');
            exit();
        }
    }
}

// Get user's division
$user_division = null;
if ($_SESSION['division_id']) {
    $user_division = $division->getDivisionById($_SESSION['division_id']);
}

// List of countries for dropdown
$countries = [
    'Afghanistan', 'Albania', 'Algeria', 'Argentina', 'Armenia', 'Australia', 'Austria', 'Azerbaijan',
    'Bangladesh', 'Belarus', 'Belgium', 'Brazil', 'Bulgaria', 'Cambodia', 'Canada', 'Chile', 'China',
    'Colombia', 'Croatia', 'Czech Republic', 'Denmark', 'Egypt', 'Estonia', 'Finland', 'France',
    'Georgia', 'Germany', 'Greece', 'Hungary', 'Iceland', 'India', 'Indonesia', 'Iran', 'Iraq',
    'Ireland', 'Israel', 'Italy', 'Japan', 'Kazakhstan', 'Kenya', 'Kuwait', 'Latvia', 'Lebanon',
    'Lithuania', 'Luxembourg', 'Malaysia', 'Mexico', 'Morocco', 'Netherlands', 'New Zealand',
    'Nigeria', 'Norway', 'Pakistan', 'Philippines', 'Poland', 'Portugal', 'Qatar', 'Romania',
    'Russia', 'Saudi Arabia', 'Singapore', 'Slovakia', 'Slovenia', 'South Africa', 'South Korea',
    'Spain', 'Sri Lanka', 'Sweden', 'Switzerland', 'Thailand', 'Turkey', 'Ukraine', 'United Arab Emirates',
    'United Kingdom', 'United States', 'Vietnam', 'Yemen', 'Zimbabwe'
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Regional Offices Form - Dashboard</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="dashboard-container">
        <!-- Sidebar -->
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>Dashboard</h2>
                <div class="user-info">
                    <div class="user-avatar"><?php echo strtoupper(substr($_SESSION['user_name'], 0, 1)); ?></div>
                    <div class="user-details">
                        <div class="user-name"><?php echo htmlspecialchars($_SESSION['user_name']); ?></div>
                        <div class="user-role"><?php echo htmlspecialchars($_SESSION['user_role']); ?></div>
                        <?php if ($user_division): ?>
                            <div class="user-division"><?php echo htmlspecialchars($user_division['name']); ?></div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            
            <nav class="sidebar-nav">
                <a href="dashboard.php" class="nav-item">
                    <i class="icon">📊</i> Dashboard
                </a>
                <a href="applications.php" class="nav-item">
                    <i class="icon">📝</i> My Applications
                </a>
                <a href="international_exhibition.php" class="nav-item">
                    <i class="icon">🌍</i> International Exhibition
                </a>
                <a href="regional_offices.php" class="nav-item active">
                    <i class="icon">🏢</i> Regional Offices
                </a>
                <a href="dashboard.php?logout=1" class="nav-item logout">
                    <i class="icon">🚪</i> Logout
                </a>
            </nav>
        </div>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="content-header">
                <h1>Regional Offices Application Form</h1>
                <div class="header-actions">
                    <a href="dashboard.php" class="btn btn-secondary">← Back to Dashboard</a>
                </div>
            </div>
            
            <div class="dashboard-cards">
                <div class="card">
                    <div class="card-header">
                        <h3>🏢 Regional Offices Application</h3>
                        <p>Submit your regional office activities and data</p>
                    </div>
                    <div class="card-content">
                        <form method="POST" class="application-form" id="regionalForm" enctype="multipart/form-data">
                            <div class="form-section">
                                <h4>Regional Information</h4>
                                
                                <div class="form-row">
                                    <div class="form-group">
                                        <label for="country">Country *</label>
                                        <select id="country" name="country" required>
                                            <option value="">Select Country</option>
                                            <?php foreach ($countries as $country): ?>
                                                <option value="<?php echo htmlspecialchars($country); ?>" 
                                                        <?php echo (isset($_POST['country']) && $_POST['country'] === $country) ? 'selected' : ''; ?>>
                                                    <?php echo htmlspecialchars($country); ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label for="produce">Produce *</label>
                                        <input type="text" id="produce" name="produce" required 
                                               placeholder="Enter produce or product name" 
                                               value="<?php echo isset($_POST['produce']) ? htmlspecialchars($_POST['produce']) : ''; ?>">
                                    </div>
                                </div>
                                
                                <div class="form-row">
                                    <div class="form-group">
                                        <label for="date_wise">Date Wise</label>
                                        <input type="date" id="date_wise" name="date_wise" 
                                               value="<?php echo isset($_POST['date_wise']) ? $_POST['date_wise'] : ''; ?>">
                                    </div>
                                    
                                    <div class="form-group">
                                        <label for="year_wise">Year Wise</label>
                                        <select id="year_wise" name="year_wise">
                                            <option value="">Select Year</option>
                                            <?php 
                                            $current_year = date('Y');
                                            for ($year = $current_year; $year >= $current_year - 10; $year--): 
                                            ?>
                                                <option value="<?php echo $year; ?>" 
                                                        <?php echo (isset($_POST['year_wise']) && $_POST['year_wise'] == $year) ? 'selected' : ''; ?>>
                                                    <?php echo $year; ?>
                                                </option>
                                            <?php endfor; ?>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="form-group">
                                    <label for="file_type">File Type *</label>
                                    <select id="file_type" name="file_type" required>
                                        <option value="">Select File Type</option>
                                        <option value="CSV" <?php echo (isset($_POST['file_type']) && $_POST['file_type'] === 'CSV') ? 'selected' : ''; ?>>CSV</option>
                                        <option value="PDF" <?php echo (isset($_POST['file_type']) && $_POST['file_type'] === 'PDF') ? 'selected' : ''; ?>>PDF</option>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label for="upload_file">File Upload *</label>
                                    <input type="file" id="upload_file" name="upload_file" required 
                                           accept=".csv,.pdf" 
                                           onchange="validateFile(this)">
                                    <small class="file-help">Accepted formats: CSV, PDF (Max size: 10MB)</small>
                                </div>
                            </div>
                            
                            <div class="form-actions">
                                <button type="submit" name="submit_regional" class="btn btn-primary">
                                    Submit Application
                                </button>
                                <button type="button" class="btn btn-secondary" onclick="resetForm()">
                                    Reset Form
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Form Guidelines -->
                <div class="card">
                    <div class="card-header">
                        <h3>📋 Form Guidelines</h3>
                    </div>
                    <div class="card-content">
                        <div class="guidelines">
                            <h4>Please ensure:</h4>
                            <ul>
                                <li>All fields marked with * are required</li>
                                <li>Select the appropriate country from the dropdown</li>
                                <li>Produce field should describe the main product or service</li>
                                <li>Either Date Wise or Year Wise should be provided (or both)</li>
                                <li>File must be in CSV or PDF format</li>
                                <li>File size should not exceed 10MB</li>
                            </ul>
                            
                            <h4>File Requirements:</h4>
                            <ul>
                                <li><strong>CSV Files:</strong> Should contain structured data with proper headers</li>
                                <li><strong>PDF Files:</strong> Should be readable and contain relevant documentation</li>
                                <li>File names should be descriptive and meaningful</li>
                            </ul>
                            
                            <h4>Division Information:</h4>
                            <?php if ($user_division): ?>
                                <p><strong>Your Division:</strong> <?php echo htmlspecialchars($user_division['name']); ?> (<?php echo htmlspecialchars($user_division['code']); ?>)</p>
                                <p><strong>Description:</strong> <?php echo htmlspecialchars($user_division['description']); ?></p>
                            <?php else: ?>
                                <p class="alert alert-error">No division assigned. Please contact administrator.</p>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="assets/js/dashboard.js"></script>
    <script>
        function resetForm() {
            if (confirm('Are you sure you want to reset the form? All data will be lost.')) {
                document.getElementById('regionalForm').reset();
            }
        }
        
        function validateFile(input) {
            const file = input.files[0];
            if (file) {
                const fileSize = file.size / 1024 / 1024; // Convert to MB
                const fileName = file.name.toLowerCase();
                const fileExtension = fileName.split('.').pop();
                
                if (!['csv', 'pdf'].includes(fileExtension)) {
                    alert('Please select a CSV or PDF file.');
                    input.value = '';
                    return;
                }
                
                if (fileSize > 10) {
                    alert('File size must be less than 10MB.');
                    input.value = '';
                    return;
                }
                
                // Update file type based on extension
                const fileTypeSelect = document.getElementById('file_type');
                fileTypeSelect.value = fileExtension.toUpperCase();
            }
        }
        
        // Form validation
        document.getElementById('regionalForm').addEventListener('submit', function(e) {
            const country = document.getElementById('country').value;
            const produce = document.getElementById('produce').value.trim();
            const fileType = document.getElementById('file_type').value;
            const uploadFile = document.getElementById('upload_file').files[0];
            const dateWise = document.getElementById('date_wise').value;
            const yearWise = document.getElementById('year_wise').value;
            
            if (!country || !produce || !fileType || !uploadFile) {
                e.preventDefault();
                alert('Please fill in all required fields and upload a file.');
                return;
            }
            
            if (!dateWise && !yearWise) {
                e.preventDefault();
                alert('Please provide either Date Wise or Year Wise information.');
                return;
            }
            
            if (produce.length < 3) {
                e.preventDefault();
                alert('Produce field must be at least 3 characters long.');
                document.getElementById('produce').focus();
                return;
            }
        });
        
        // Auto-populate file type based on file selection
        document.getElementById('upload_file').addEventListener('change', function() {
            const file = this.files[0];
            if (file) {
                const fileName = file.name.toLowerCase();
                const fileExtension = fileName.split('.').pop();
                if (['csv', 'pdf'].includes(fileExtension)) {
                    document.getElementById('file_type').value = fileExtension.toUpperCase();
                }
            }
        });
    </script>
</body>
</html>
